﻿namespace YourKit.Demo;

using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Runtime.CompilerServices;
using System.Threading;
using System;


internal static class Util {
  public static bool isPrime(long n) {
    for (long i = 2; i < n / 2; ++i) {
      if (n % i == 0) {
        return false;
      }
    }
    return true;
  }
}

internal class Job {
  private long myRangeStart;
  private long myRangeEnd;
  public List<long> myPrimes = [];

  public Job(long start, long end) {
    myRangeStart = start;
    myRangeEnd = end;
  }

  public void run() {
    for (long n = myRangeStart; n < myRangeEnd; n++) {
      if (Util.isPrime(n)) {
        myPrimes.Add(n);
      }
    }
  }
}

internal class Worker {
  private readonly BlockingCollection<Job> myJobs;

  public Worker(BlockingCollection<Job> jobs) {
    myJobs = jobs;
  }

  public void processJobs() {
    while (!myJobs.IsCompleted) {
      try {
        Job job = myJobs.Take();
        job.run();
      }
      catch (InvalidOperationException) {
      }
    }
  }
}

internal class CpuDemo {
  public void run() {
    var workerCount = Math.Max(Environment.ProcessorCount / 4, 1);
    var jobs = new BlockingCollection<Job>(workerCount);

    for (int i = 1; i <= workerCount; i++) {
      new Thread(new Worker(jobs).processJobs) { Name = "Worker #" + i}.Start();
    }

    const int workSize = 1000;
    for (long n = 0; n < long.MaxValue; n += workSize) {
      jobs.Add(new Job(n, n + workSize));
    }
    jobs.CompleteAdding();
  }
}


internal class GcDemo {
  private readonly int[] myPinnedArray = GC.AllocateArray<int>(10000, pinned: true);

  public void run() {
    while (true) {
      GC.Collect();
      Thread.Sleep(TimeSpan.FromSeconds(20));
    }
  }
}

internal class ExceptionsDemo {
  public void run() {
    var random = new Random();
    while (true) {
      try {
        generateException(random);
      }
      catch (DivideByZeroException) {
      }
    }
  }

  [MethodImpl(MethodImplOptions.NoInlining | MethodImplOptions.NoOptimization)]
  private int generateException(Random random) {
    return divideByZero(random);
  }

  [MethodImpl(MethodImplOptions.NoInlining | MethodImplOptions.NoOptimization)]
  private int divideByZero(Random random) {
    return random.Next(100) / random.Next(100);
  }
}

internal class Progress {
  public void run() {
    while (true) {
      Console.Write(".");
      Thread.Sleep(TimeSpan.FromSeconds(1));
    }
  }
}

public class Program {
  public static void Main() {
    new Thread(new CpuDemo().run) { Name = "CPU demo thread"}.Start();
    new Thread(new GcDemo().run) { Name = "GC demo thread"}.Start();
    new Thread(new ExceptionsDemo().run) { Name = "Exceptions demo thread"}.Start();

    Console.WriteLine("The demo application is running a simulation.");
    Console.WriteLine("Connect to the application from the profiler UI to see profiling results.");
    Console.WriteLine("Press Enter to exit the application.");

    new Thread(new Progress().run) { Name = "Progress thread"}.Start();

    Console.ReadLine();
    Environment.Exit(0);
  }
}
